package net.lecnam.info.util;

import java.util.AbstractList;
import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Supplier;

public abstract class ImmutableList<A> extends AbstractList<A> {

	public static <A> Boolean isEmpty(List<A> l) {
		return l.size() == 0;
	}

	public static <A> int size(List<A> l) {
		return l.size();
	}

	public static <A> A hd(List<A> l) {
		return l.get(0);
	}

	public static <A> List<A> tl(List<A> l) {
		return l.subList(1, l.size());
	}

	public static <A> List<A> concat(List<A> l1, List<A> l2) {
		if (l1.isEmpty())
			return l2;
		else
			return cons(hd(l1), concat(tl(l1), l2));
	}

	public static <A, R> R match(List<A> l, Supplier<R> f1,
			BiFunction<A, List<A>, R> f2) {
		if (l.isEmpty())
			return f1.get();
		else
			return f2.apply(hd(l), tl(l));
	}

	public static <A> ImmutableList<A> fromList(List<A> l) {
		if (l.isEmpty())
			return new Nil<>();
		else
			return new Cons<A>(hd(l), tl(l));
	}

	public static <A> List<A> nil() {
		return new Nil<A>();
	}

	public static <A> List<A> cons(A head, List<A> tail) {
		return new Cons<A>(head, tail);
	}

}

class Nil<A> extends ImmutableList<A> {
	public Nil() {
	}

	@Override
	public A get(int index) {
		throw new IndexOutOfBoundsException();
	}

	@Override
	public int size() {
		return 0;
	}
}

class Cons<A> extends ImmutableList<A> {
	final A head;
	final List<A> tail;

	public Cons(A head, List<A> tail) {
		this.head = head;
		this.tail = tail;
	}

	public A hd() {
		return head;
	}

	public List<A> tl() {
		return tail;
	}

	@Override
	public A get(int index) {
		if (index == 0)
			return head;
		else
			return tail.get(index - 1);
	}

	@Override
	public int size() {
		return 1 + tail.size();
	}

	@Override
	public List<A> subList(int fromIndex, int toIndex) {
		if (fromIndex == 1 && toIndex == size())
			return tail;
		else
			return super.subList(fromIndex, toIndex);
	}

}
