package net.lecnam.info.ex3;

import static net.lecnam.info.ex3.TokenSeqs.locToken;
import static net.lecnam.info.ex3.Tokens.symbols;
import static net.lecnam.info.util.Functions.apply;
import static net.lecnam.info.util.ImmutableList.concat;
import static net.lecnam.info.util.ImmutableList.cons;
import static net.lecnam.info.util.ImmutableList.hd;
import static net.lecnam.info.util.ImmutableList.nil;
import static net.lecnam.info.util.ImmutableList.tl;
import static net.lecnam.info.util.Pair.pair;

import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Reader;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;

import net.lecnam.info.util.Pair;

public class Lexer {


    public static <B, A> Optional<B> find(List<Pair<A, B>> l, A x) {
        if (l.isEmpty()) {
            return Optional.empty();
        } else {
            var h = hd(l);
            var t = tl(l);
            return h.match((k, v) -> {
                if (Objects.equals(x, k)) {
                    return Optional.of(v);
                } else {
                    return find(t, x);
                }
            });
        }
    }

    public static <B, A> List<B> map(Function<A, B> f, List<A> l) {
        if (l.isEmpty()) {
            return nil();
        } else {
            var h = hd(l);
            var t = tl(l);
            return cons(apply(f, h), map(f, t));
        }
    }

    public static String implode(List<Character> l) {
        if (l.isEmpty()) {
            return "";
        } else {
            var h = hd(l);
            var t = tl(l);
            return (String.valueOf(h) + implode(t));
        }
    }

    public static List<Character> explodeRec(String s, Integer i) {
        if (i < s.length()) {
            return cons(s.charAt(i), explodeRec(s, (i + 1)));
        } else {
            return nil();
        }
    }

    public static List<Character> explode(String s) {
        return explodeRec(s, 0);
    }

    public static <A> Boolean isPrefix(List<A> l1, List<A> l2) {
        if (l1.isEmpty()) {
            return true;
        } else {
            var h1 = hd(l1);
            var t1 = tl(l1);
            if (l2.isEmpty()) {
                return false;
            } else {
                var h2 = hd(l2);
                var t2 = tl(l2);
                if (Objects.equals(h1, h2)) {
                    return isPrefix(t1, t2);
                } else {
                    return false;
                }
            }
        }
    }

    public static List<Character> findSymbol(List<Pair<String, Token>> sl, List<Character> l) {
        if (sl.isEmpty()) {
            throw new NoSuchElementException();
        } else {
            var h = hd(sl);
            var tl = tl(sl);
            return h.match((s, _1) -> {
                var cl = explode(s);
                if (isPrefix(cl, l)) {
                    return cl;
                } else {
                    return findSymbol(tl, l);
                }
            });
        }
    }

    public static <A> List<A> discard(Integer n, List<A> l) {
        if (l.isEmpty()) {
            return nil();
        } else {
            var t = tl(l);
            if (Objects.equals(n, 0)) {
                return l;
            } else {
                return discard((n - 1), t);
            }
        }
    }

    public static Pair<List<Character>, List<Character>> cutSymbol(List<Character> l) {
        try {
            var cl = findSymbol(symbols, l);
            var n = cl.size();
            return pair(cl, discard(n, l));
        } catch (NoSuchElementException _exn) {
            return pair(List.of(hd(l)), tl(l));
        }
    }

    public static Token makeSymbol(String s) {
        var opt = find(symbols, s);
        if (opt.isEmpty()) {
            return new T_UNKNOWN();
        } else {
            var tok = opt.get();
            return tok;
        }
    }

    public static <A> Pair<List<A>, List<A>> splitlRec(Predicate<A> p, List<A> l, List<A> l2) {
        if (l2.isEmpty()) {
            return pair(l, nil());
        } else {
            var x = hd(l2);
            var tl = tl(l2);
            if (apply(p, x)) {
                return splitlRec(p, concat(l, List.of(x)), tl);
            } else {
                return pair(l, l2);
            }
        }
    }

    public static <A> Pair<List<A>, List<A>> splitl(Predicate<A> p, List<A> l) {
        return splitlRec(p, nil(), l);
    }

    public static <A> Boolean member(A x, List<A> l) {
        if (l.isEmpty()) {
            return false;
        } else {
            var h = hd(l);
            var t = tl(l);
            if (Objects.equals(x, h)) {
                return true;
            } else {
                return member(x, t);
            }
        }
    }
    public static Function<Integer, Character> chr = (Integer i) -> { int c = i; return (char)c; };
    public static Function<Character, Integer> code = (Character c) -> { char i = c; return (int)i; };
    public static Predicate<Character> isLetter = (Character c) -> {
            return ((!(c < 'a')) && (!(c > 'z')));
        };
    public static Predicate<Character> isUpper = (Character c) -> {
            return ((!(c < 'A')) && (!(c > 'Z')));
        };
    public static Predicate<Character> isDigit = (Character c) -> {
            return ((!(c < '0')) && (!(c > '9')));
        };
    public static Predicate<Character> isPunct = (Character c) -> {
            return (((!(c < '!')) && (!(c > '~'))) || Objects.equals(c, '&'));
        };
    public static Predicate<Character> isAlpha = (Character c) -> {
            return (apply(isLetter, c) || apply(isUpper, c));
        };
    public static Predicate<Character> isAlphaNum = (Character c) -> {
            return (Objects.equals(c, '_') || (apply(isAlpha, c) || apply(isDigit, c)));
        };
    public static Predicate<Character> isNewline = (Character c) -> {
            return Objects.equals(apply(code, c), 10);
        };
    public static Predicate<Character> isDblQuote = (Character c) -> {
            return Objects.equals(c, '"');
        };
    public static Predicate<Character> noDblQuote = (Character c) -> {
            return (!Objects.equals(c, '"'));
        };

    public static Token scanString(List<Character> l) {
        return new T_String(implode(l));
    }

    public static Token scanNum(List<Character> l) {
        return new T_Int(Integer.parseInt(implode(l)));
    }

    public static Token scanSymb(List<Character> l) {
        return makeSymbol(implode(l));
    }

    public static Token scanAlphaNum(List<Character> l) {
        var s = implode(l);
        return new T_Ident(s);
    }

    public static List<Pair<Token, Integer>> scan(Integer line, List<Character> s) {
        if (s.isEmpty()) {
            return nil();
        } else {
            var c = hd(s);
            var s1 = tl(s);
            if (apply(isDblQuote, c)) {
                var p = splitl(noDblQuote, s1);
                return p.match((l, s2) -> {
                    return cons(pair(scanString(l), line), scan(line, tl(s2)));
                });
            } else {
                if (apply(isAlpha, c)) {
                    var p = splitl(isAlphaNum, s);
                    return p.match((l, s2) -> {
                        return cons(pair(scanAlphaNum(l), line), scan(line, s2));
                    });
                } else {
                    if (apply(isDigit, c)) {
                        var p = splitl(isDigit, s);
                        return p.match((l, s2) -> {
                            return cons(pair(scanNum(l), line), scan(line, s2));
                        });
                    } else {
                        if (apply(isPunct, c)) {
                            var p = cutSymbol(s);
                            return p.match((l, s2) -> {
                                return cons(pair(scanSymb(l), line), scan(line, s2));
                            });
                        } else {
                            if (apply(isNewline, c)) {
                                return scan((line + 1), s1);
                            } else {
                                return scan(line, s1);
                            }
                        }
                    }
                }
            }
        }
    }

    public static List<Character> readAll(Reader ic) {
        try {
            var b = ic.read();
            if (b < 0) {
                return nil();
            } else {
                return cons(apply(chr, b), readAll(ic));
            }
        } catch (IOException _exn) {
            return nil();
        }
    }

    public static List<Character> readLinesRec(Reader ic, Boolean linebreak) {
        try {
            var b = ic.read();
            if (b < 0) {
                return nil();
            } else {
                var c = apply(chr, b);
                var newline = apply(isNewline, c);
                if (linebreak && newline) {
                    return nil();
                } else {
                    return cons(c, readLinesRec(ic, newline));
                }
            }
        } catch (IOException _exn) {
            return nil();
        }
    }

    public static List<Character> readLines(Reader ic) {
        return readLinesRec(ic, false);
    }

    public static List<LocToken> lexer(List<Character> s) {
        return map((Pair<Token, Integer> p) -> {
            return locToken(p);
        }, scan(1, s));
    }

    public static List<LocToken> lexFile(String filename) {
        try {
            var reader = new FileReader(filename);
            var lines = readLines(reader);
            var tokens = lexer(lines);
            return tokens;
        } catch (Exception msg) {
            throw new Error(msg);
        }
    }

    public static void println(PrintWriter oc, String s) {
        oc.print(s);
        oc.print("\n");
        oc.flush();
        return;
    }

    public static void run(List<String> args) {
        var filename = (Objects.equals(args, nil())) ? "src/main/resources/test.json" : hd(args);
        var result = lexFile(filename);
        System.out.print(result);
        return;
    }

    public static void main(String[] args) {
        run(List.of(args));
    }
}

