package net.lecnam.info.ex1;

import static net.lecnam.info.ex1.Lexer.readLines;
import static net.lecnam.info.ex1.TokenSeqs.getFirstLine;
import static net.lecnam.info.ex1.TokenSeqs.makeSeq;
import static net.lecnam.info.ex1.TokenSeqs.peek;
import static net.lecnam.info.ex1.TokenSeqs.remove;
import static net.lecnam.info.util.ImmutableList.cons;
import static net.lecnam.info.util.ImmutableList.hd;
import static net.lecnam.info.util.ImmutableList.nil;

import java.io.FileReader;
import java.io.Reader;
import java.util.List;
import java.util.Objects;

public class Parser {


    public static void parse(TokenSeq s, Token t) {
        var h = remove(s);
        if (Objects.equals(h.tok(), t)) {
            return;
        } else {
            throw new Error("Unexpected token line " + String.valueOf(h.line()));
        }
    }

    public static String parseString(TokenSeq s) {
        var h = remove(s);
        switch (h.tok()) {
            case T_String(var name) -> {
                return name;
            }
            default -> {
                throw new Error("String expected line " + String.valueOf(h.line()));
            }
        }
    }

    public static Integer parseInt(TokenSeq s) {
        var h = remove(s);
        switch (h.tok()) {
            case T_Int(var i) -> {
                return i;
            }
            default -> {
                throw new Error("Integer expected line " + String.valueOf(h.line()));
            }
        }
    }

    public static Expr parseExpr(TokenSeq s) {
        parse(s, new T_LCURL());
        parse(s, new T_String("@type"));
        parse(s, new T_COLON());
        var name = parseString(s);
        switch (name) {
            case "Const" -> {
                parse(s, new T_COMMA());
                parse(s, new T_String("i"));
                parse(s, new T_COLON());
                var i = parseInt(s);
                parse(s, new T_RCURL());
                return new Const(i);
            }
            case "Minus" -> {
                parse(s, new T_COMMA());
                parse(s, new T_String("e"));
                parse(s, new T_COLON());
                var e1 = parseExpr(s);
                parse(s, new T_RCURL());
                return new Minus(e1);
            }
            case "Plus" -> {
                parse(s, new T_COMMA());
                parse(s, new T_String("e1"));
                parse(s, new T_COLON());
                var e1 = parseExpr(s);
                parse(s, new T_COMMA());
                parse(s, new T_String("e2"));
                parse(s, new T_COLON());
                var e2 = parseExpr(s);
                parse(s, new T_RCURL());
                return new Plus(e1, e2);
            }
            case "Times" -> {
                parse(s, new T_COMMA());
                parseString(s);
                parse(s, new T_COLON());
                var e1 = parseExpr(s);
                parse(s, new T_COMMA());
                parseString(s);
                parse(s, new T_COLON());
                var e2 = parseExpr(s);
                parse(s, new T_RCURL());
                return new Times(e1, e2);
            }
            case "Avg" -> {
                parse(s, new T_COMMA());
                parseString(s);
                parse(s, new T_COLON());
                var el = parseExprList(s);
                return new Avg(el);
            }
            default -> {
                throw new Error("Unknown record name line " + String.valueOf(getFirstLine(s)));
            }
        }
    }

    public static List<Expr> parseExprList(TokenSeq s) {
        parse(s, new T_LSQUARE());
        var h = peek(s);
        switch (h.tok()) {
            case T_RSQUARE() -> {
                remove(s);
                return nil();
            }
            default -> {
                var e = parseExpr(s);
                var l = parseOtherExprs(s);
                return cons(e, l);
            }
        }
    }

    public static List<Expr> parseOtherExprs(TokenSeq s) {
        var h = peek(s);
        switch (h.tok()) {
            case T_RSQUARE() -> {
                remove(s);
                return nil();
            }
            case T_COMMA() -> {
                remove(s);
                var e = parseExpr(s);
                var l = parseOtherExprs(s);
                return cons(e, l);
            }
            default -> {
                throw new Error("Syntax error line " + String.valueOf(getFirstLine(s)));
            }
        }
    }

    public static Expr parseExprWithReader(Reader reader) {
        try {
            var content = readLines(reader);
            var tokens = makeSeq(Lexer.lexer(content));
            return parseExpr(tokens);
        } catch (Exception err) {
            throw new Error(err);
        }
    }

    public static Expr parseExprFromFile(String filename) {
        try {
            var reader = new FileReader(filename);
            return parseExprWithReader(reader);
        } catch (Exception err) {
            throw new Error(err);
        }
    }

    public static void run(List<String> args) {
        var filename = (Objects.equals(args, nil())) ? "src/main/resources/test.json" : hd(args);
        var result = parseExprFromFile(filename);
        System.out.println(result);
        System.out.println(Pretty.stringOf(result)); 
        return;
    }

    public static void main(String[] args) {
        run(List.of(args));
    }
}

