package net.lecnam.info;

import java.util.AbstractList;

//sealed interface SimpleList<A> {}
//record Nil<A>() implements SimpleList<A> {}
//record Cons<A>(A head, SimpleList<A> tail) implements SimpleList<A> {}

sealed abstract class ImmutableList<A> extends AbstractList<A> {

	@Override
	public abstract int size();

	@Override
	public abstract A get(int index);

	public interface ListVisitor<R, A> {
        public R visitNil();
        public R visitCons(A h, ImmutableList<A> t);
	}

	public abstract <R> R accept(ListVisitor<R, A> v);

	public static <A> ImmutableList<A> nil() {
		return new Nil<>();
	}

	public static <A> ImmutableList<A> cons(A h, ImmutableList<A> t) {
		return new Cons<>(h, t);
	}

	public abstract A hd();
	public abstract ImmutableList<A> tl();

    @SafeVarargs
	public static <A> ImmutableList<A> of(A... elements) {
        ImmutableList<A> l = nil();
        for (var i = elements.length - 1; i >= 0; i--) {
            l = cons(elements[i], l);
        }
        return l;
    }

}

final class Nil<A> extends ImmutableList<A> {

	public Nil() {
		super();
	}

	@Override
	public int size() {
		return 0;
	}

	@Override
	public A get(int index) {
		throw new IndexOutOfBoundsException();
	}

	@Override
	public <R> R accept(ListVisitor<R, A> v) {
		return v.visitNil();
	}

	@Override
	public A hd() {
		throw new IndexOutOfBoundsException();
	}

	@Override
	public ImmutableList<A> tl() {
		throw new IndexOutOfBoundsException();
	}

}

final class Cons<A> extends ImmutableList<A> {

	final private A head;
	final private ImmutableList<A> tail;

	public Cons(A head, ImmutableList<A> tail) {
		super();
		this.head = head;
		this.tail = tail;
	}

	@Override
	public int size() {
		return 1 + tail.size();
	}

	@Override
	public A get(int index) {
		if (index < 0 || index >= size()) {
			throw new IndexOutOfBoundsException();
		}
		if (index == 0) {
			return head;
		} else {
			return tail.get(index - 1);
		}
	}

	@Override
	public <R> R accept(ListVisitor<R, A> v) {
		return v.visitCons(head, tail);
	}

	@Override
	public A hd() {
		return head;
	}

	@Override
	public ImmutableList<A> tl() {
		return tail;
	}

}
